/**
* @package com_spbooking
* @author JoomShaper http://www.joomshaper.com
* @copyright Copyright (c) 2010 - 2023 JoomShaper
* @license http://www.gnu.org/licenses/gpl-2.0.html GNU/GPLv2 or later
*/

;(function($, window, document, undefined) {
    var name = "spGallery";

    function Plugin(element, options) {
        this.element    = element;
        this.name       = name;
        this._defaults  = $.fn.spGallery.defaults;
        this.options    = $.extend({}, this._defaults, options);
        this.uploadProgress = [];
        this.levelUpPath = '';
        this.selectedFiles = [];
        
        this.options.mediaLibrary = $.extend({}, this.options.mediaLibrary, this.options.media);

        this.currentDirectory = this.options.mediaLibrary.rootPath;
        
        // Call the init function
        this.init();
    }

    
    $.extend(Plugin.prototype, {
        init: function() {
            this.cacheElement();
            this.createHTMLDOM();
            this.dragAndDropHandlers(this.$spgallery_drop_area, '', 0);
            this.createMediaLibraryHTMLDOM();
            this.renderFolderList();
            
        },
        
        cacheElement: function() {
            this.$element = $(this.element);
        },
        initProgress: function(numberOfFiles) {
            this.uploadProgress = [];
            this.$upload_progress.val(0);
            if (this.$upload_progress.hasClass('completed')) {
                this.$upload_progress.removeClass('completed');
            }
            for(i = 0; i < numberOfFiles; i++) {
                this.uploadProgress.push(0);
            }
        },

        updateProgress: function(fileIndex, percent) {
            this.uploadProgress[fileIndex] = percent;
            let total = this.uploadProgress.reduce(function(tot, curr){
                return tot + curr;
            }) / this.uploadProgress.length;
            this.$upload_progress.val(total);

        },

        createDOMElement: function(tagName, attr = {}, innerHTML = '') {
            tagName     = typeof(tagName) === 'string' && tagName.length > 0 ? tagName : false;
            attr        = typeof(attr) === 'object' && Object.keys(attr).length ? attr : false;
            innerHTML   = typeof(innerHTML) === 'string' && innerHTML.length > 0 ? innerHTML : false;

            let _element;
            
            // IF tag name given then create element, otherwise create element div
            if (tagName) {
                _element = document.createElement(tagName);
            } else {
                _element = document.createElement("div");
            }

            // If attributes given
            if (attr) {
                Object.keys(attr).forEach(function(key) {
                    _element.setAttribute(key, attr[key]);
                });
            }

            // Add inner HTML
            if (innerHTML) {
                _element.innerHTML = innerHTML;
            }

            return _element;
        },
        
        createHTMLDOM: function() {
            let self = this;

            // Create spgallery-drop-area element
            // @parent this.$element
            let _spgallery_drop_area = this.createDOMElement('div', {
                'class': 'spgallery-drop-area',
                'id': 'spgallery-drop-area-' + this.options.spgalleryId
            });
            this.$element.append(_spgallery_drop_area);
            this.$spgallery_drop_area = $(_spgallery_drop_area);
            
            // Create spgallery-form element 
            // @parent spgallery-drop-area
            let _spgallery_form = this.createDOMElement('div', {
                'class': 'spgallery-form'
            });
            this.$spgallery_drop_area.append(_spgallery_form);
            this.$spgallery_form = $(_spgallery_form);

            // Create button-container
            // @parent spgallery-form
            let _button_container = this.createDOMElement('div', {
                'class': 'button-container'
            });

            this.$spgallery_form.append(_button_container);
            this.$button_container = $(_button_container);

            // Create upload-image button
            // @parent button-container
            let _upload_image = this.createDOMElement('button', {
                'type': 'button',
                'class': 'spgallery-btn btn-select-images upload-image is-primary'
            }, "<span class='fa fa-upload'></span> <span>" + this.options.uploadImageBtnText + "</span>");
            this.$button_container.append(_upload_image);
            this.$upload_image = $(_upload_image);
    
            // Create btn-open-media button
            // @parent button-container
            let _btn_open_media = this.createDOMElement('button', {
                'type': 'button',
                'class': 'spgallery-btn btn-open-media is-primary'
            }, "<span class='fa fa-picture-o'></span> <span>" + this.options.openMediaBtnText + "</span>");
            this.$button_container.append(_btn_open_media);
            this.$btn_open_media = $(_btn_open_media);
            _btn_open_media.onclick = this.openModal();

            // Create drop-here text
            // @parent spgallery-form
            let _drop_here = this.createDOMElement('p',{}, this.options.dropHereText);
            this.$spgallery_form.append(_drop_here);

            // Create underline
            // @parent spgallery-form
            let _underline = this.createDOMElement('span',{'class': 'underline'});
            this.$spgallery_form.append(_underline);
            
            // Create progress-wrapper 
            // @parent spgallery-form
            let _progress_wrapper = this.createDOMElement('div',{
                'class': 'progress-wrapper'
            });
            this.$progress_wrapper = $(_progress_wrapper);
            this.$spgallery_form.append(_progress_wrapper);

            // Create progress-bar element
            // @parent progress-wrapper
            let _upload_progress = this.createDOMElement('progress', {
                'class': 'progress-bar',
                'min': 0,
                'max': 100,
                'value': 0
            });
            this.$upload_progress = $(_upload_progress);
            this.$progress_wrapper.append(_upload_progress);
            



            // Create fileElem input element
            // @parent spgallery-form
            let _fileElem = this.createDOMElement('input', {
                'type': 'file',
                'class': 'fileElem',
                'id': 'fileElem_' + this.options.spgalleryId,
                'multiple': 'multiple',
                'accept': 'image/*'
            });
            
            this.$fileElem = $(_fileElem);
            
            // file click event on fileElem when click upload image button
            this.$upload_image.on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.$fileElem.click();
            });

            // handle all files onchange the file input
            this.$fileElem.on('change', function(e) {
                self.uploadHandler(this.files,'',0);
            });

            // Create gallery element
            // @parent spgallery-drop-area
            let _gallery = this.createDOMElement('div', {
                'class': 'gallery'
            });
            this.$spgallery_drop_area.append(_gallery);
            this.$gallery = $(_gallery);

            // Create input field element
            // @parent spgallery-drop-area
            let _input_field = this.createDOMElement('input', {
                'type': 'hidden',
                'name': this.options.fieldName,
                'id': this.options.spgalleryId,
                'value': this.options.fieldValue
            });
            this.$spgallery_drop_area.append(_input_field);
            this.$input_field = $(_input_field);

            this.createValueHTMLDOM(this.options.fieldValue, this.$gallery);
        },

        openModal: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                if (!self.$sp_modal.hasClass('sp-modal-show')) {
                    self.$sp_modal.addClass('sp-modal-show');
                    self.refreshMediaBody();
                }
            }
        },
        
        closeModalOnclick: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                self.closeModal();
            }
        },

        closeModal: function() {
            if (this.$sp_modal.hasClass('sp-modal-show')) {
                this.$sp_modal.removeClass('sp-modal-show');
            }
        },

        renderImageFile(image, savedFlag = 1) {
            image = typeof(image) == 'object' && Object.keys(image).length > 0 ? image : false;

            // Create image-wrapper element
            // @parent @parent
            let saved_class = (savedFlag == 1) ? 'saved-src' : '';
            let _image_wrapper = this.createDOMElement('div', {
                'class': 'image-wrapper ' + saved_class
            });
            $image_wrapper = $(_image_wrapper);

            // Create image element
            // @parent image-wrapper
            let _image = this.createDOMElement('img', {
                'src': self.options.baseUrl + image.image,
                'alt': image.alt_text
            });
            $image_wrapper.append(_image);
            $image = $(_image);

            // Create remove-image-wrapper element
            // @parent image-wrapper
            let _remove_image_wrapper = this.createDOMElement('div', {
                'class': 'remove-image-wrapper'
            });
            $image_wrapper.append(_remove_image_wrapper);
            $remove_image_wrapper = $(_remove_image_wrapper);
            
            // Create cross-holder element
            // @parent remove-image-wrapper
            let _cross_holder = this.createDOMElement('div', {
                'class': 'cross-holder'
            });
            $remove_image_wrapper.append(_cross_holder);
            $cross_holder = $(_cross_holder);
            
            // Create remove-image element
            // @parent cross-holder
            let _remove_image = this.createDOMElement('a', {
                'class': 'remove-image spgallery-close',
                'href': 'javascript:',
                'data-src': image.image
            });

            _remove_image.onclick = this.removeImage();
            
            $cross_holder.append(_remove_image);
            $remove_image = $(_remove_image);

            return _image_wrapper;
        },

        createValueHTMLDOM: function(value, parent) {
            value = typeof(value) === 'string' && value.length > 0 ? JSON.parse(value) : false;
            self  = this;
            this.$wrapper = [];

            if (value) {
                Object.keys(value).forEach(function(key){
                    let image = {
                        'image': value[key].image,
                        'alt_text': value[key].alt_text
                    };
                    let img_element = self.renderImageFile(image, 1);
                    parent.append(img_element);
                    self.$wrapper.push($(img_element));
                });
            }
        },

        removeImage: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                let val = typeof(self.$input_field.val()) == 'string' && self.$input_field.val().length > 0 ? self.$input_field.val() : false;
                if (val) {
                    val = typeof(val) == 'string' && val.length > 0 ? JSON.parse(val) : val;
                    let imgSrc = typeof($(this).data('src')) == 'string' && $(this).data('src').length > 0 ? $(this).data('src') : false;
                    if (Object.keys(val).length && imgSrc) {
                        let result = {};
                        Object.keys(val).map(function(key) {
                            if (val[key].image != imgSrc) {
                                result[key] = val[key];
                            }
                        });
                        result = typeof(result) == 'object' && Object.keys(result).length > 0 ? JSON.stringify(result) : '';
                        self.$input_field.val(result);
                    }
                }
                // Remove the image but not from the server
                $(this).closest('.image-wrapper').remove();
                
                if ($('.image-wrapper').length <= 0) {
                    self.$upload_progress.css('visibility', 'hidden');
                }
            }
        },

        dragAndDropHandlers: function(element, path, flag) {
            let self = this;

            // Reset drag events
            $("body").off('dragenter dragover dragleave drop');
            element.off('dragenter dragover dragleave drop');
            
            // Prevent defaults
            $("body").on('dragenter dragover dragleave drop', function(e) {
                e.preventDefault();
                e.stopPropagation();
            });
            // Prevent defaults
            element.on('dragenter dragover dragleave drop', function(e) {
                e.preventDefault();
                e.stopPropagation();
            });
            
            // Highlight on dragenter and dragover
            element.on('dragenter dragover', function(e) {
                e.preventDefault();
                e.stopPropagation();
                if (flag == 0) {
                    $(this).addClass('highlight');
                } else if (flag == 1) {
                    $(this).addClass('dragging');
                }
                $(this).addClass('highlight');
            });

            // Unhighlight on dragleave and drop
            element.on('dragleave drop', function(e) {
                e.preventDefault();
                e.stopPropagation();
                if (flag == 0) {
                    $(this).removeClass('highlight');
                } else if (flag == 1) {
                    $(this).removeClass('dragging');
                }
                
            });

            // Drop handler
            element.on('drop', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.dropHandler(e, path, flag);
            });
        },

        dropHandler: function(e, path, flag) {
            let files       = [];
            let dt          = e.originalEvent.dataTransfer;
            let tempFiles   = dt.files;
            files           = [...tempFiles];
            
            if (files.length > 0) {
                this.$loading.show();
            }
            this.initProgress(files.length);
            this.uploadHandler(files, path, flag);
        },

        uploadHandler: function(files, path, flag) {
            files = typeof(files) == 'object' && files instanceof Array ? files : [...files];
            let self = this;
            files.map(function(file, index) {
                self.uploadFile(file, index, path, flag);
            })
        },

        uploadFile: function(file, index, path, flag) {
            let url = typeof(this.options.host) == 'string' && this.options.host.length > 0 ? this.options.host + '&task=spgallery.upload' : false;
            path = typeof(path) == 'string' && path.length > 0 ? path : this.options.uploadPath;
            flag = typeof(flag) == 'number' && typeof(flag) != 'undefined' ? flag : 0;

            let self = this;
            
            if (url) {
                let formData = new FormData();
                formData.append('file', file);
                formData.append('uploadPath', btoa(path));
                $.ajax({
                    xhr: function() {
                        let xhr = new window.XMLHttpRequest();
                        xhr.upload.addEventListener('progress', function(event){
                            if (event.lengthComputable) {
                                let percentCompleted = event.loaded / event.total;
                                percentCompleted = parseInt(percentCompleted * 100);
                                let progressValue = (event.loaded * 100.0 / event.totla) || 100;
                                if (flag == 0) {
                                    self.$upload_progress.css('visibility', 'visible');
                                    self.updateProgress(index, progressValue);
                                }

                                if (percentCompleted === 100) {
                                    if (flag == 0) {
                                        self.$upload_progress.addClass('completed');
                                    }
                                }
                            }
                        }, false);
                        return xhr;
                    },
                    type: 'POST',
                    url: url,
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        response = typeof(response) == 'string' && response.length > 0 ? JSON.parse(response) : false;
                        if (response) {
                            if (response.status_code == '200') {
                                if (flag == 0) {
                                    let obj = {
                                        'image': atob(response.path),
                                        'alt_text': response.alt_text
                                    };
                                    let input = typeof(self.$input_field.val()) == 'string' && self.$input_field.val().length > 0 ? JSON.parse(self.$input_field.val()) : false;
                                    self.updateInputField(input, obj);
                                } else if (flag == 1) {
                                    self.refreshMediaBody();
                                }
                            } else {
                                console.error("Error uploading images");
                            }
                        }
                    },
                    complete: function(){
                        
                    },
                    error: function(err) {
                        console.error(err);
                    }
                })
            } else {
                console.error("Unrecognized URL or url not provided");
            }
        },

        updateInputField: function(input, item) {
            if (input) {
                // concat with previous string
                let max = 0;
                Object.keys(input).forEach(function(key) {
                    if (parseInt(key.replace(/^[^0-9]+/, ''), 10) > max) {
                        max = parseInt(key.replace(/^[^0-9]+/, ''), 10);
                    }
                });
                let index = this.options.indexName + (max+1);
                input[index] = item;
                console.log('upInput', JSON.stringify(input));
                this.$input_field.val(JSON.stringify(input));
                
            } else {
                // add new string
                let index = this.options.indexName + '1';
                let temp = {};
                temp[index] = item;
                if (temp) {
                    this.$input_field.val(JSON.stringify(temp));
                }
            }

            // Attach the image to the board
            let img_element = this.renderImageFile(item, 0);
            this.$gallery.append(img_element);
            this.$wrapper.push($(img_element));
        },


        /**
         * Media Libraray functions
         */

        createMediaLibraryHTMLDOM: function() {
            // Create sp-modal element
            // @parent spgallery-drop-area
            let _sp_modal = this.createDOMElement('div', {
                'class': 'sp-modal'
            });
            this.$spgallery_drop_area.append(_sp_modal);
            this.$sp_modal = $(_sp_modal);
            
            // Create sp-modal-container element
            // @parent sp-modal
            let _sp_modal_container = this.createDOMElement('div', {
                'class': 'sp-modal-container'
            });
            this.$sp_modal.append(_sp_modal_container);
            this.$sp_modal_container = $(_sp_modal_container);
            
            // Create sp-modal-header element
            // @parent sp-modal-container
            let _sp_modal_header = this.createDOMElement('div', {
                'class': 'sp-modal-header'
            });
            this.$sp_modal_container.append(_sp_modal_header);
            this.$sp_modal_header = $(_sp_modal_header);
            
            // Create sp-modal-header-h4 element
            // @parent sp-modal-header
            let _sp_modal_header_h4 = this.createDOMElement('h4', {}, this.options.mediaLibrary.titleText);
            this.$sp_modal_header.append(_sp_modal_header_h4);
            this.$sp_modal_header_h4 = $(_sp_modal_header_h4);

            // Create sp-modal-close element
            // @parent sp-modal-header
            let _sp_modal_close = this.createDOMElement('div', {
                'class': 'sp-modal-close'
            }, "<span class='fa fa-times'></span>");
            this.$sp_modal_header.append(_sp_modal_close);
            this.$sp_modal_close = $(_sp_modal_close);
            _sp_modal_close.onclick = this.closeModalOnclick();
            
            // Create sp-modal-body element
            // @parent sp-modal-container
            let _sp_modal_body = this.createDOMElement('div',{
                'class': 'sp-modal-body'
            });
            this.$sp_modal_container.append(_sp_modal_body);
            this.$sp_modal_body = $(_sp_modal_body);

            // MOdal body contents
            let _modal_body_contents = this.renderMediaBody();
            this.$sp_modal_body.append(_modal_body_contents);
            this.$modal_body_contents = $(_modal_body_contents);
            
            // Create sp-modal-footer element
            // @parent sp-modal-container
            let _sp_modal_footer = this.createDOMElement('div',{
                'class': 'sp-modal-footer'
            });
            this.$sp_modal_container.append(_sp_modal_footer);
            this.$sp_modal_footer = $(_sp_modal_footer);
            
            // Create close-modal element
            // @parent sp-modal-footer
            let _close_modal = this.createDOMElement('button',{
                'class': 'spgallery-btn close-modal',
                'type': 'button'
            }, this.options.mediaLibrary.modalCloseBtn);
            this.$sp_modal_footer.append(_close_modal);
            this.$close_modal = $(_close_modal);
            _close_modal.onclick = this.closeModalOnclick();
            
        },

        insertMedia: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                files = typeof(self.selectedFiles) == 'object' && self.selectedFiles instanceof Array && self.selectedFiles.length > 0 ? self.selectedFiles : false;
                if (files) {
                    files.map(function(file) {
                        let input = typeof(self.$input_field.val()) == 'string' && self.$input_field.val().length > 0 ? JSON.parse(self.$input_field.val()) : false;
                        let obj = {
                            'image': file,
                            'alt_text': file
                        };
                        self.updateInputField(input, obj);
                    });
                }
                self.selectedFiles = [];
                self.closeModal();
            }
        },

        renderMediaBody: function() {
            // Create spgallery-media-container element
            // @parent none | return this
            let _spgallery_media_container = this.createDOMElement('div',{
                'class': 'spgallery-media-container'
            });
            $media_container = $(_spgallery_media_container);
            
            // Create spgallery-media-heading element
            // @parent spgallery-media-container
            let _spgallery_media_heading = this.createDOMElement('div',{
                'class': 'spgallery-media-heading'
            });
            $media_container.append(_spgallery_media_heading);
            this.$spgallery_media_heading = $(_spgallery_media_heading);

            // Create media-heading-left
            // @parent spgallery-media-heading
            let _media_heading_left = this.createDOMElement('div', {
                'class': 'media-heading-left'
            });
            this.$spgallery_media_heading.append(_media_heading_left);
            this.$media_heading_left = $(_media_heading_left);
            
            // Create media-heading-right
            // @parent spgallery-media-heading
            let _media_heading_right = this.createDOMElement('div', {
                'class': 'media-heading-right'
            });
            this.$spgallery_media_heading.append(_media_heading_right);
            this.$media_heading_right = $(_media_heading_right);

            // Create ok-modal | insert-media element
            // @parent sp-modal-footer
            let _ok_modal = this.createDOMElement('button',{
                'class': 'spgallery-btn ok-modal is-success insert-media',
                'type': 'button',
                'disabled': 'disabled'
            }, "<span class='fa fa-check'></span> <span>" + this.options.mediaLibrary.modalOkBtn + "</span>");
            _ok_modal.onclick = this.insertMedia();
            this.$media_heading_right.append(_ok_modal);
            this.$ok_modal = $(_ok_modal);

            // Create media-remove element
            // @parent spgallery-media-heading
            let _media_remove = this.createDOMElement('button', {
                'class': 'spgallery-btn media-remove is-danger',
                'type': 'button',
                'disabled': 'disabled'
            }, "<span class='fa fa-trash'></span> <span>"+ this.options.mediaLibrary.modalRemoveBtn + "</span>");
            
            this.$media_heading_right.append(_media_remove);
            this.$media_remove = $(_media_remove);
            _media_remove.onclick = this.removeSelected();


            // Left header buttons

            // Create new-folder button
            // @parent media-heading-left
            let _new_folder = this.createDOMElement('button', {
                'class': 'new-folder spgallery-btn is-primary',
                'type': 'button'
            }, "<span class='fa fa-plus'></span> New Folder");

            this.$media_heading_left.append(_new_folder);
            this.$new_folder = $(_new_folder);
            _new_folder.onclick = this.newFolder();

            // Create folder-list select element
            // @parent media-heading-left
            let _folder_list = this.createDOMElement('select', {
                'class': 'folder-list spgallery-btn is-success',
                'id': 'folder-list'
            });

            this.$media_heading_left.append(_folder_list);
            this.$folder_list = $(_folder_list);
            _folder_list.onchange = this.goToFolder();

             // Create media-up element
            // @parent spgallery-media-heading
            let _media_up = this.createDOMElement('button',{
                'class': 'spgallery-btn media-up is-primary',
                'type': 'button',
                'disabled': 'disabled'
            }, "<span class='fa fa-level-up'></span> <span>"+ this.options.mediaLibrary.modalDirectoryUp +"</span>");
            
            this.$media_heading_left.append(_media_up);
            this.$media_up = $(_media_up);
            _media_up.onclick = this.levelUp();

            let _loading = this.createDOMElement('div', {
                'class': 'spinner'
            },"<div class='spinner-icon'></div>");
            this.$media_heading_left.append(_loading);
            this.$loading = $(_loading);
            

            // Create information-container element
            // @parent spgallery-media-container
            let _information_container = this.createDOMElement('div', {
                'class': 'information-container'
            });
            $media_container.append(_information_container);
            this.$information_container = $(_information_container);

            // Create media-navigator-holder element
            // @parent spgallery-media-container
            let _media_navigator_holder = this.createDOMElement('ul', {
                'class': 'media-navigator-holder spbreadcrumb'
            });
            this.$information_container.append(_media_navigator_holder);
            this.$media_navigator_holder = $(_media_navigator_holder);

            // Create selection-count element
            // @parent information-container
            let _selection_holder = this.createDOMElement('div', {
                'class': 'selection-count'
            });
            this.$information_container.append(_selection_holder);
            this.$selection_holder = $(_selection_holder);
            
            // Create spgallery-media-body element
            // @parent spgallery-media-container
            let _spgallery_media_body = this.createDOMElement('div',{
                'class': 'spgallery-media-body'
            });
            $media_container.append(_spgallery_media_body);
            this.$spgallery_media_body = $(_spgallery_media_body);

            // Create drag-message element
            // @parent spgallery-media-body
            let _drag_message = this.createDOMElement('p',{
                'class': 'drag-message'
            },'DROP HERE');
            this.$spgallery_media_body.append(_drag_message);


            // Create files-container element
            // @parent spgallery-media-body
            let _files_container = this.createDOMElement('div', {
                'class': 'files-container'
            });
            this.$files_container = $(_files_container);

            this.fetchDirectoryContent(this.options.mediaLibrary.rootPath).then(function(content) {
                self.renderContents(content);
            });
            

            return _spgallery_media_container;
        },

        renderFolderList: function() {
            let self = this;
            
            this.folderTree(this.options.mediaLibrary.rootPath).then(function(response) {
                response = typeof(response) == 'string' && response.length > 0 ? JSON.parse(response) : false;
                if (response) {
                    self.$folder_list.empty();
                    let _first_option = self.createDOMElement('option', {
                        'value': ''
                    }, '-- Jump Into Directory --');
                    self.$folder_list.append(_first_option);
                    
                    response.map(function(dir) {
                        dir = dir.replace(/^\/|\/$/g, '');
                        let _option = self.createDOMElement('option', {
                            'value': dir
                        }, dir);
                        if (dir == self.currentDirectory) {
                            _option.setAttribute('selected', 'selected');
                        }
                        self.$folder_list.append(_option);
                    });
                }
            });
        },

        folderTree: function(path) {
            let self = this;
            return new Promise(function(resolve, reject) {
                $.ajax({
                    url: self.options.host + "&task=spgallery.getAllPossibleDirectories",
                    type: 'POST', 
                    data: {
                        'path': btoa(path)
                    },
                    success: function(response) {
                        resolve(response);
                    },
                    error: function(err) {
                        reject(err);
                    }
                });
            });
        },

        newFolder: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                let directory = self.currentDirectory;
                let folderName = prompt("Enter the folder name");
                if (folderName) {
                    $.ajax({
                        url: self.options.host + "&task=spgallery.newFolder",
                        type: 'POST',
                        data: {
                            'directory': btoa(directory + '/' + folderName)
                        },
                        success: function(response) {
                            if (response == 200) {
                                self.reloadMediaBody();   
                            }
                        }
                    });
                }
            }
        },

        removeSelected: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                e.stopPropagation();
                let confirm = window.confirm('The selected file(s) will be removed parmanently. Do you really want to perform this task?');
                if (confirm) {
                    self.removeFilesFromServer(self.selectedFiles).then(function(response) {
                        self.reloadMediaBody();
                    });
                }
            }
        },

        removeFilesFromServer: function(files) {
            let self = this;
            return new Promise(function(resolve, reject) {
                files = typeof(files) == 'object' && files instanceof Array && files.length > 0 ? JSON.stringify(files) : false;
                if (files) {
                    
                    $.ajax({
                        url: self.options.host + '&task=spgallery.removeFiles',
                        data: {
                            'files': btoa(files)
                        },
                        type: 'POST',
                        success: function(response) {
                            resolve(response);
                        },
                        error: function(err) {
                            reject(err);
                        }
                    });
                } else {
                    reject('No file');
                }
            });
        },

        updateNavigator: function() {
            this.$media_navigator_holder.empty();
            
            // Create media-navigator element
            // @parent media-navigator-holder
            let navInfo = this.currentDirectory;
            navInfo = navInfo.split('/');

            
            let self = this;
            if (navInfo.length > 0) {
                let _media_navigator = [];
                navInfo.map(function(val, index) {
                    _navigator = self.createDOMElement('li', {}, "<a href='javascript:'>" +(val)+ "</a>");
                    _media_navigator[index] = _navigator;
                    _media_navigator[index].onclick = self.diveDeepFolders(navInfo.slice(0,index+1).join('/'));
                    self.$media_navigator_holder.append(_media_navigator);
                });
            }
        },

        updateSelectedImageCount: function() {
            this.$selection_holder.empty();
            if (this.selectedFiles.length) {
                let _selection_count = this.createDOMElement('span', {}, "Image Selected: " + this.selectedFiles.length);
                this.$selection_holder.append(_selection_count);
            }
        },

        reloadMediaBody: function() {
            let self = this;
            path = this.currentDirectory;
            path = typeof(path) == 'string' && path.length > 0 ? path : false;
            if (path) {
                this.fetchDirectoryContent(path).then(function(content) {
                    self.renderContents(content);
                });
                let way = path.split('/');
                way.pop();
                self.levelUpPath = way.join('/');
                self.currentDirectory = way.join('/');
            }
        },

        refreshMediaBody: function() {
            let self = this;
            path = this.currentDirectory;
            path = typeof(path) == 'string' && path.length > 0 ? path : false;
            if (path) {
                this.fetchDirectoryContent(path).then(function(content) {
                    self.renderContents(content);
                    self.$loading.hide();
                });
            }
        },
        
        levelUp: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                e.stopPropagation();
                path = self.levelUpPath;
                path = typeof(path) == 'string' && path.length > 0 ? path : false;
                if (path) {
                    self.fetchDirectoryContent(path).then(function(content) {
                        self.renderContents(content);
                    });
                    let way = path.split('/');
                    way.pop();
                    self.levelUpPath = way.join('/');
                    self.currentDirectory = way.join('/');
                }
            }
        },

        fetchDirectoryContent: function(path) {
            let self = this;
            return new Promise(function(resolve, reject) {
                path = typeof(path) == 'string' && path.length > 0 ? path : 'images';
                if (path) {
                    $.ajax({
                        url: self.options.host + '&task=spgallery.getDirectoryContents',
                        type: 'POST',
                        data: {
                            'path': btoa(path)
                        },
                        success: function(response) {
                            resolve(response);
                        },
                        error: function(err) {
                            reject(err);
                        }
                    });
                }
            });
            
        },
        
        diveDeepFolders: function(path) {
            let self = this;
            return function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.fetchDirectoryContent(path).then(function(content) {
                    self.renderContents(content);
                    let way = path.split('/');
                    way.pop();
                    self.levelUpPath = way.join('/');
                }).catch(function(err){
                    alert("The directory has already been removed from the disk!");
                });
                
            }
        },

        goToFolder: function() {
            let self = this;
            return function(e) {
                e.preventDefault();
                e.stopPropagation();
                let path = $(this).val();
                self.fetchDirectoryContent(path).then(function(content) {
                    self.renderContents(content);
                    let way = path.split('/');
                    way.pop();
                    self.levelUpPath = way.join('/');
                }).catch(function(err){
                    alert("The directory has already been removed from the disk!");
                });

            }
        },

        renderContents: function(content) {
            //@TODO if deepSelection enabled then not emptying this.slectedFiles
            this.selectedFiles = [];
            this.changeDisabledState(this.$ok_modal);
            this.changeDisabledState(this.$media_remove);
            this.updateSelectedImageCount();
            this.renderFolderList();

            let self = this;
            content  = typeof(content) == 'string' && content.length > 0 ? JSON.parse(content) : false;
            if (content) {
                self.currentDirectory = content.path;
                self.dragAndDropHandlers(self.$spgallery_media_body, self.currentDirectory, 1);
                // update navigator
                self.updateNavigator();

                self.$spgallery_media_body.empty();

                content.folders.map(function(folder) {
                    let _folder = self.renderFolder(folder.name, content.path);
                    self.$spgallery_media_body.append(_folder);
                });

                content.files.map(function(file) {
                    let _files = self.renderFile(file, content.path);
                    self.$spgallery_media_body.append(_files);
                });
            }
            
            // update disabled state of level up button
            let currentDir = this.currentDirectory.split('/');
            if (currentDir.length > 1) {
                this.$media_up.prop('disabled', false);
            } else {
                this.$media_up.prop('disabled', true);
            }
            
        },

        renderFolder: function(folder, path) {
            // Create folder-wrapper element
            let _folder_wrapper = this.createDOMElement('div', {
                'class': 'folder-wrapper is-folder',
                'data-path': path + '/' + folder,
                'title': folder
            });
            $folder_wrapper = $(_folder_wrapper);
            _folder_wrapper.onclick = this.diveDeepFolders(path + '/' + folder);


            // Create folder-link element
            // @parent folder-wrapper
            let _folder_link = this.createDOMElement('a', {
                'class': 'folder-link',
                'href': 'javascript:'
            }, "<span class='fa fa-folder fa-5x'></span><p>"+folder+"</p>");
            $folder_wrapper.append(_folder_link);
            
            return _folder_wrapper;
        },

        renderFile: function(file, path) {
            // Create folder-wrapper element
            let _folder_wrapper = this.createDOMElement('div', {
                'class': 'folder-wrapper is-file',
                'data-filePath': path+ '/' + file
            });
            _folder_wrapper.onclick = this.selectThis(path + '/' + file);
            $folder_wrapper = $(_folder_wrapper);


            let _image = this.createDOMElement('img', {
                'class': 'is-responsive',
                'src': this.options.baseUrl + path + '/' + file
            });
            $folder_wrapper.append(_image);
            $image = $(_image);

            let _image_title = this.createDOMElement('p', {
                'class': 'image-title',
                'title': file
            }, file);

            $folder_wrapper.append(_image_title);
            $image_title = $(_image_title);

            return _folder_wrapper;
        },

        selectThis: function(filePath) {
            let self = this;
            return function(e) {
                e.preventDefault();
                e.stopPropagation();

                $this = $(this);
                if ($this.hasClass('selected')) {
                    $this.removeClass('selected');
                    self.selectedFiles = self.selectedFiles.removeElement(filePath);
                } else {
                    $this.addClass('selected');
                    self.selectedFiles.push(filePath);
                }
                self.changeDisabledState(self.$media_remove);
                self.changeDisabledState(self.$ok_modal);
                self.updateSelectedImageCount();
            }
        },

        changeDisabledState: function(element) {
            if (this.selectedFiles.length > 0 ) {
                element.prop('disabled', false)
            } else {
                element.prop('disabled',true);
            }
        }


    });

    Array.prototype.removeElement = function(item) {
        let _arr = [];
        this.map(function(value) {
            if (value != item) {
                _arr.push(value);
            }
        });
        return _arr;
    }

    $.fn.spGallery = function(options) {
        this.each(function() {
            if (!$.data(this, name)) {
                $.data(this, name, new Plugin(this, options));
            }
        });
        return this;
    };

    $.fn.spGallery.defaults = {
        baseUrl: '',
        host: '',
        uploadPath: 'images/component/gallery',
        indexName: 'spgallery',
        spgalleryId: 'spgallery',
        fieldName: 'spgallery',
        fieldValue: [],
        uploadImageBtnText: 'Upload',
        openMediaBtnText: 'Media Library',
        dropHereText: 'OR DROP HERE',
        media: {},
        mediaLibrary: {
            titleText: 'Media Library',
            modalCloseBtn: 'Close',
            modalOkBtn: 'Insert',
            modalRemoveBtn: 'Remove',
            modalDirectoryUp: 'Up',
            rootPath: 'images'

            
        }
    };
}) (jQuery, window, document);